#!/usr/bin/env python

import os, platform, re, shutil, sys

SUPPORTED_BLENDER_VERSIONS = ['3.4', '3.5', '3.6', '4.0', '4.1', '4.2', '4.3', '4.4', '4.5']

def copyAddon(blendConfDir, asExtension):

    rootDir = os.path.normpath(os.path.join(os.path.dirname(os.path.abspath(__file__)), '..', '..', '..'))

    srcPath = os.path.join(rootDir, 'addons', 'verge3d')

    if asExtension:
        dstParentPath = os.path.join(blendConfDir, 'extensions', 'soft8soft')
    else:
        dstParentPath = os.path.join(blendConfDir, 'scripts', 'addons')

    os.makedirs(dstParentPath, exist_ok=True)

    dstPath = os.path.join(dstParentPath, 'verge3d')

    if os.path.isdir(dstPath):
        shutil.rmtree(dstPath)

    shutil.copytree(srcPath, dstPath)

    # set root path in addon init script

    srcInitScript = os.path.join(srcPath, '__init__.py')
    dstInitScript = os.path.join(dstPath, '__init__.py')

    with open(srcInitScript, 'r', encoding='utf-8') as fin:
        with open(dstInitScript, 'w', encoding='utf-8') as fout:
            for line in fin:
                fout.write(re.sub('(ROOT_DIR) *=.*', 'ROOT_DIR = r\'{}\''.format(rootDir.replace('\\', '\\\\')), line))

    if asExtension:
        startupPath = os.path.join(blendConfDir, 'scripts', 'startup')
        os.makedirs(startupPath, exist_ok=True)

        shutil.copy2(os.path.join(srcPath, 'setup', 'startup.py'), os.path.join(startupPath, 'verge3d.py'))
        shutil.copy2(os.path.join(srcPath, 'setup', 'startup_migrate.py'), os.path.join(startupPath, 'soft8soft_repo_migrate.py'))

def removeAddon(blendConfDir, asExtension):

    removed = False

    if asExtension:
        path = os.path.join(blendConfDir, 'extensions', 'soft8soft', 'verge3d')
    else:
        path = os.path.join(blendConfDir, 'scripts', 'addons', 'verge3d')

    if os.path.isdir(path):
        shutil.rmtree(path)
        removed = True

    if asExtension:
        startupScript = os.path.join(blendConfDir, 'scripts', 'startup', 'verge3d.py')
        if os.path.isfile(startupScript):
            os.remove(startupScript)
            removed = True
        migrateScript = os.path.join(blendConfDir, 'scripts', 'startup', 'soft8soft_repo_migrate.py')
        if os.path.isfile(migrateScript):
            os.remove(migrateScript)
            removed = True

    return removed

def traverseBlenderDirs(doInstall=True):
    system = platform.system()

    if system == 'Windows':
        blendDir = os.path.expandvars(r'%USERPROFILE%\AppData\Roaming\Blender Foundation\Blender')
    elif system == 'Darwin':
        blendDir = os.path.expandvars(r'$HOME/Library/Application Support/Blender')
    else:
        blendDir = os.path.expandvars(r'$HOME/.config/blender')

    for blendVer in SUPPORTED_BLENDER_VERSIONS:
        blendVerDir = os.path.join(blendDir, blendVer)
        asExtension = (float(blendVer) >= 4.2)

        if doInstall:
            print('Installing Verge3D {} for Blender {}'.format('extension' if asExtension else 'addon', blendVer))
            copyAddon(blendVerDir, asExtension)

            # COMPAT: remove legacy addon from Blender 4.2+, if installed
            if asExtension:
                removeAddon(blendVerDir, False)
        else:
            if removeAddon(blendVerDir, asExtension):
                print('Removed Verge3D {} for Blender {}'.format('extension' if asExtension else 'addon', blendVer))

if __name__ == '__main__':

    if len(sys.argv) <= 1 or sys.argv[1].upper() == 'INSTALL':
        traverseBlenderDirs(True)
    elif len(sys.argv) > 1 and sys.argv[1].upper() == 'UNINSTALL':
        traverseBlenderDirs(False)
    else:
        print('Wrong script arguments')
